﻿using System;
using System.Collections.Generic;
using System.Text;
using System.Threading;

namespace MicroRWD.Common
{
    // Abstract generic reader
    public abstract class Reader : IDisposable
    {
        #region Private Constants

        // The tag status command (ASCII 'S')
        private const byte C_CMD_CARD_STATUS = 0x53;

        // The message command (ASCII 'z')
        private const byte C_CMD_MESSAGE = 0x7a;

        // The program EEPROM command (ASCII 'P')
        private const byte C_CMD_PROGRAM_EEPROM = 0x50;

        // The factory reset command (ASCII 'F')
        private const byte C_CMD_FACTORY_RESET = 0x46;

        // Poll Interval (in milliseconds)
        private const int C_POLL_INTERVAL_MS = 1000;

        #endregion


        #region Private Properties

        // Port handler to manage underlying port
        private PortHandler PortHandler { get; set; }

        // Port handler mutex to guard against concurrent access
        private object PortLock { get; set; }

        // Reader thread
        private Thread ReaderThread { get; set; }

        // Flag to stop reader thread
        private bool Stopped { get; set; }

        // Reader acknowledge event handler
        private event AcknowledgeEventHandler acknowledgeEvent;

        #endregion


        #region Public Properties

        // Command response timeout in milliseconds
        public int CommandTimeoutMillis { get; set; }

        // Continuous poll
        public bool ContinuousPoll { get; set; }

        // Optional polling function to be used instead of Status()
        private Func<object> pollingFunction = null;
        public Func<object> PollingFunction
        {
            get { return pollingFunction; }
            set { pollingFunction = value; }
        }

        // Port name (e.g. "COM1")
        public string PortName
        {
            get
            {
                // Delegate to port handler
                return PortHandler.PortName;
            }

            set
            {
                // Only act on changes
                if (!String.Equals(PortHandler.PortName, value))
                {
                    // Gain exclusive access to port handler
                    lock (PortLock)
                    {
                        // Release port handler resources
                        PortHandler.Dispose();

                        // Create port handler to manage underlying port
                        PortHandler = new PortHandler(value);
                    }
                }
            }
        }

        // Port state
        public PortState PortState { get { return PortHandler.PortState; } }

        #endregion

        // Status update event handler
        public event AcknowledgeEventHandler AcknowledgeEvent { add { acknowledgeEvent += value; } remove { acknowledgeEvent -= value; } }

        #region Constructor

        // Constructs a new MicroRWD object attached via the specified port name
        public Reader(string portName, int commandTimeoutMillis)
        {
            // Create port handler to manage underlying port
            PortHandler = new PortHandler(portName);

            // Create port handler mutex to guard against concurrent access
            PortLock = new object();

            // Initialise command response timeout (in milliseconds)
            CommandTimeoutMillis = commandTimeoutMillis;

            // Initialise continuous poll
            ContinuousPoll = true;

            // Initialise stopped flag
            Stopped = true;
        }

        #endregion


        #region Public Methods

        // Stop the background polling thread
        public void Start()
        {
            // Prevent concurrent access to internal state
            lock (this)
            {
                if (Stopped)
                {
                    // Start the reader thread
                    Stopped = false;
                    ReaderThread = new Thread(ReaderRun);
                    ReaderThread.Start();
                }
                else
                {
                    Log.Error("already started");
                }
            }
        }

        // Stop the background polling thread
        public void Stop()
        {
            // Prevent concurrent access to internal state
            lock (this)
            {
                if (!Stopped)
                {
                    // Stop the reader thread
                    Stopped = true;
                    ReaderThread.Join();
                }
                else
                {
                    Log.Error("already stopped");
                }
            }
        }

        // Factory Reset
        public void CmdFactoryReset()
        {
            // Placeholder for the reply (don't expect one)
            byte[] reply = new byte[0];

            // Send the request, awaiting response or timeout
            CmdGeneric(new byte[] { C_CMD_FACTORY_RESET, 0x55, 0xAA });
        }

        // Product and Firmware Identifier Message
        public string CmdMessage()
        {
            // Message if a reply is received, empty string otherwise
            string result = "";

            // Send the request, awaiting response or timeout
            byte[] reply = CmdGeneric(new byte[] { C_CMD_MESSAGE });

            // If we get a response convert it to an ASCII string
            if (reply.Length > 0)
            {
                // Decode result
                result = ASCIIEncoding.ASCII.GetString(reply);
            }

            // Return result
            return result;
        }

        #region IDisposable Methods

        // Dispose of the underlying resources
        public void Dispose()
        {
            // Shutdown the reader if not stopped
            if (!Stopped)
            {
                Stop();
            }

            // Dispose of the resources allocated by the port handler
            PortHandler.Dispose();

            // Prevent further finalisation
            GC.SuppressFinalize(this);
        }

        #endregion

        #endregion


        #region Protected Methods

        // Generic command
        protected byte[] CmdGeneric(byte[] _command)
        {
            // Zero length reply on timeout, data otherwise
            byte[] result = new byte[0];

            // Ignore null or empty command
            if ((_command != null) && (_command.Length > 0))
            {
                // Gain exclusive access to port handler
                lock (PortLock)
                {
                    // Send request via port handler, awaiting response or timeout
                    result = PortHandler.Request(_command, CommandTimeoutMillis);

                    // Check for reply
                    if (result.Length > 0)
                    {
                        // Fire acknowledge event except for commands with no acknowledgement!
                        if ((_command[0] != 0x46) && (_command[0] != 0x7a))
                        {
                            FireAcknowledgeEvent(result[0]);
                        }
                    }
                    else
                    {
                        // Fire a blank acknoledge event to clear the status
                        FireAcknowledgeEvent(0);
                    }
                }
            }
            else
            {
                // Error
                Log.Error("null or empty command");
            }

            // Return result
            return result;
        }

        // Program EEPROM
        protected byte CmdProgram(byte location, byte value)
        {
            // Set if a reply is received, 0x00 otherwise
            byte result = 0x00;

            // Send the request, awaiting response or timeout
            byte[] reply = CmdGeneric(new byte[] { C_CMD_PROGRAM_EEPROM, location, value });

            // If we get a response extract acknowledge byte
            if (reply.Length == 1)
            {
                // Extract acknowledgement
                result = reply[0];
            }

            // Return result
            return result;
        }

        // Status
        protected byte CmdStatus()
        {
            // Set if a reply is received, 0x00 otherwise
            byte result = 0x00;

            // Send the request, awaiting response or timeout
            byte[] reply = CmdGeneric(new byte[] { C_CMD_CARD_STATUS });

            // If we get a response extract acknowledge byte
            if (reply.Length == 1)
            {
                // Extract acknowledgement
                result = reply[0];
            }

            // Return result
            return result;
        }

        #endregion


        #region Private Methods

        // Fire acknowledge event to listeners
        private void FireAcknowledgeEvent(byte _value)
        {
            try
            {
                // Instantiate suitable event args object
                AcknowledgeEventArgs args = new AcknowledgeEventArgs(_value);

                // Check for current listeners
                if (acknowledgeEvent != null)
                {
                    // Dispatch event to listeners
                    acknowledgeEvent(this, args);
                }
            }
            catch (Exception ex)
            {
                Log.Error(ex.ToString());
            }
        }

        private void ReaderRun()
        {
            // Information
            Log.Information("begin");

            // Now repeat until stopped
            while (!Stopped)
            {
                // Check state of continuous poll
                if (ContinuousPoll)
                {
                    if (PollingFunction != null)
                    {
                        // Poll using custom polling function
                        PollingFunction();
                    }
                    else
                    {
                        // Perform regular status byte check
                        CmdStatus();
                    }
                }

                // Wait until it is time for the next poll
                Thread.Sleep(C_POLL_INTERVAL_MS);
            }

            // Information
            Log.Information("end");
        }

        #endregion
    }
}
